import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { useRouter } from "next/router";
import { useLanguage } from "@/contexts/LanguageContext";
import { BarChart3, CheckCircle2, Zap, AlertTriangle } from "lucide-react";
import TaskOverviewTab from "@/components/analysis/tasks/TaskOverviewTab";
import TaskCompletionTab from "@/components/analysis/tasks/TaskCompletionTab";
import TaskPerformanceTab from "@/components/analysis/tasks/TaskPerformanceTab";
import TaskNonConformityTab from "@/components/analysis/tasks/TaskNonConformityTab";

type TaskTabId = "overview" | "completion" | "performance" | "nonConformity";

type TaskAnalysisData = {
  taskStats: {
    totalTasks: number;
    completedTasks: number;
    pendingTasks: number;
    overdueTasks: number;
    recurringTasks: number;
    nonRecurringTasks: number;
    periodCompletionRate: number;
    totalPeriods: number;
    completedPeriods: number;
    tasksCreatedToday: number;
    tasksCreatedThisWeek: number;
    tasksCreatedThisMonth: number;
  };
  patternStats: Array<{
    pattern: string;
    total: number;
    completed: number;
    completionRate: number;
  }>;
  departmentStats: Array<{
    department: string;
    total: number;
    completed: number;
    overdue: number;
    nonConformities: number;
    completionRate: number;
  }>;
  technicianStats: Array<{
    technicianId: string;
    technicianName: string;
    technicianEmail: string;
    isExternal: boolean;
    companyName: string | null;
    total: number;
    completed: number;
    overdue: number;
    avgCompletionTime: number;
    completionRate: number;
  }>;
  nonConformityStats: {
    totalNonConformities: number;
    tasksWithNonConformities: number;
    nonConformityRate: number;
  };
};


export default function TaskAnalysis() {
  const { data: session, status } = useSession();
  const { t } = useLanguage();
  const router = useRouter();
  const [taskData, setTaskData] = useState<TaskAnalysisData | null>(null);
  const [loading, setLoading] = useState(true);
  const [activeTab, setActiveTab] = useState<TaskTabId>("overview");

  useEffect(() => {
    if (status === "loading") return;
    if (!session || (session.user as any)?.role !== "ADMIN") {
      router.push("/");
      return;
    }
    fetchData();
  }, [session, status, router]);

  const fetchData = async () => {
    try {
      setLoading(true);
      const response = await fetch("/api/admin/task-analysis");
      if (response.ok) {
        const data = await response.json();
        setTaskData(data);
      }
    } catch (error) {
      console.error("Error fetching task analysis:", error);
    } finally {
      setLoading(false);
    }
  };

  if (status === "loading" || loading) {
    return (
      <div className='min-h-screen bg-gray-50'>
        <div className='container mx-auto px-4 py-8'>
          <div className='text-center'>{t("common.loading") || "Loading..."}</div>
        </div>
      </div>
    );
  }

  return (
    <div className='min-h-screen bg-gray-50'>
      <div className='container mx-auto px-4 sm:px-6 py-6 sm:py-8'>
        <div className='mb-6 sm:mb-8'>
          <h1 className='text-2xl sm:text-3xl font-bold text-gray-900 mb-2'>
            {t("analysis.taskAnalysis.title") || "Task Analysis"}
          </h1>
          <p className='text-sm sm:text-base text-gray-600'>
            {t("analysis.taskAnalysis.subtitle") ||
              "Comprehensive analysis of tasks, completion rates, performance, and non-conformities"}
          </p>
        </div>

        {/* Tab Navigation */}
        <div className='mb-6 sm:mb-8'>
          <div className='border-b border-gray-200 overflow-x-auto overflow-y-hidden'>
            <nav className='flex space-x-2 sm:space-x-4 md:space-x-8 -mb-px min-w-fit'>
              {[
                { id: "overview", icon: BarChart3 },
                { id: "completion", icon: CheckCircle2 },
                { id: "performance", icon: Zap },
                { id: "nonConformity", icon: AlertTriangle },
              ].map((tab) => (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id as TaskTabId)}
                  className={`py-2 px-2 sm:px-3 md:px-1 border-b-2 font-medium text-xs sm:text-sm flex items-center whitespace-nowrap flex-shrink-0 ${
                    activeTab === tab.id
                      ? "border-blue-500 text-blue-600"
                      : "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
                  }`}
                >
                  <tab.icon className='w-3 h-3 sm:w-4 sm:h-4 mr-1 sm:mr-2 flex-shrink-0' />
                  <span className='hidden sm:inline'>
                    {t(`analysis.taskAnalysis.tabs.${tab.id}`) || tab.id}
                  </span>
                  <span className='sm:hidden'>
                    {t(`analysis.taskAnalysis.tabs.${tab.id}Short`) || tab.id}
                  </span>
                </button>
              ))}
            </nav>
          </div>
        </div>

        {/* Tab Content */}
        {activeTab === "overview" && taskData && (
          <TaskOverviewTab taskData={taskData} />
        )}

        {activeTab === "completion" && taskData && (
          <TaskCompletionTab taskData={taskData} />
        )}

        {activeTab === "performance" && taskData && (
          <TaskPerformanceTab taskData={taskData} />
        )}

        {activeTab === "nonConformity" && taskData && (
          <TaskNonConformityTab taskData={taskData} />
        )}
      </div>
    </div>
  );
}

